"use strict";
/**
 * Auto-Updater for RepairIQ Sync Agent v1.0.9
 *
 * Enterprise-grade auto-update using Windows Scheduled Task:
 * - Downloads update ZIP to fixed location
 * - Verifies SHA256 hash
 * - Triggers scheduled task (runs as SYSTEM)
 * - Task has permission to stop/start services
 * - Automatic rollback on failure
 *
 * Architecture:
 * 1. Service detects update available
 * 2. Service downloads ZIP and verifies hash
 * 3. Service triggers "RepairIQUpdater" scheduled task
 * 4. Task stops service, updates files, starts service
 */
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getCurrentVersion = getCurrentVersion;
exports.startAutoUpdater = startAutoUpdater;
const fs = __importStar(require("fs"));
const path = __importStar(require("path"));
const crypto = __importStar(require("crypto"));
const node_fetch_1 = __importDefault(require("node-fetch"));
const logger_1 = require("./logger");
const child_process_1 = require("child_process");
// Configuration
const UPDATE_CHECK_INTERVAL = 6 * 60 * 60 * 1000; // 6 hours
const MAX_RETRY_ATTEMPTS = 3;
const UPDATES_DIR = 'C:\\RepairIQ-EMS\\updates';
const UPDATE_ZIP_PATH = path.join(UPDATES_DIR, 'update.zip');
const UPDATE_INFO_PATH = path.join(UPDATES_DIR, 'update.json');
const SCHEDULED_TASK_NAME = 'RepairIQUpdater';
/**
 * Get version file path (in install directory)
 */
function getVersionFilePath() {
    // When running as service, we're in C:\Program Files\RepairIQ\dist
    // So go up one level and store version.json there
    return path.join(path.resolve(__dirname, '..'), 'version.json');
}
/**
 * Get current local version - ALWAYS reads from package.json (source of truth)
 */
function getCurrentVersion() {
    // ALWAYS read from package.json - this is the source of truth
    // version.json is only used for tracking failed update attempts, NOT version
    try {
        const packageJsonPath = path.join(path.resolve(__dirname, '..'), 'package.json');
        if (fs.existsSync(packageJsonPath)) {
            const packageJson = JSON.parse(fs.readFileSync(packageJsonPath, 'utf-8'));
            return packageJson.version || '1.0.0';
        }
    }
    catch (error) {
        (0, logger_1.log)(`Error reading package.json: ${error}`, 'WARN');
    }
    return '1.0.0';
}
/**
 * Save current version
 */
function saveVersion(version, failedAttempts = 0) {
    try {
        const versionFile = getVersionFilePath();
        const versionData = {
            version,
            lastCheckDate: new Date().toISOString(),
            failedAttempts,
        };
        fs.writeFileSync(versionFile, JSON.stringify(versionData, null, 2));
        (0, logger_1.log)(`Saved version ${version} (attempts: ${failedAttempts})`);
    }
    catch (error) {
        (0, logger_1.log)(`Error saving version file: ${error}`, 'ERROR');
    }
}
/**
 * Semantic version comparison
 */
function compareVersions(v1, v2) {
    const parts1 = v1.split('.').map(Number);
    const parts2 = v2.split('.').map(Number);
    for (let i = 0; i < Math.max(parts1.length, parts2.length); i++) {
        const p1 = parts1[i] || 0;
        const p2 = parts2[i] || 0;
        if (p1 > p2)
            return 1;
        if (p1 < p2)
            return -1;
    }
    return 0;
}
/**
 * Get failed attempt count
 */
function getFailedAttempts() {
    try {
        const versionFile = getVersionFilePath();
        if (fs.existsSync(versionFile)) {
            const data = JSON.parse(fs.readFileSync(versionFile, 'utf-8'));
            return data.failedAttempts || 0;
        }
    }
    catch (error) {
        (0, logger_1.log)(`Error reading failed attempts: ${error}`, 'WARN');
    }
    return 0;
}
/**
 * Increment failed attempts
 */
function incrementFailedAttempts() {
    const currentVersion = getCurrentVersion();
    const attempts = getFailedAttempts() + 1;
    saveVersion(currentVersion, attempts);
    return attempts;
}
/**
 * Reset failed attempts (called after successful update)
 */
function resetFailedAttempts() {
    const currentVersion = getCurrentVersion();
    saveVersion(currentVersion, 0);
}
/**
 * Ensure updates directory exists
 */
function ensureUpdatesDir() {
    if (!fs.existsSync(UPDATES_DIR)) {
        fs.mkdirSync(UPDATES_DIR, { recursive: true });
        (0, logger_1.log)(`Created updates directory: ${UPDATES_DIR}`);
    }
}
/**
 * Calculate SHA256 hash of a file
 */
function calculateFileHash(filePath) {
    const fileBuffer = fs.readFileSync(filePath);
    const hash = crypto.createHash('sha256');
    hash.update(fileBuffer);
    return hash.digest('hex');
}
/**
 * Check if an update is available
 */
async function checkForUpdate(apiUrl) {
    try {
        (0, logger_1.log)('Checking for updates...');
        const response = await (0, node_fetch_1.default)(`${apiUrl}/api/sync-agent/version`, {
            method: 'GET',
            redirect: 'follow',
            headers: { 'Content-Type': 'application/json' },
            timeout: 30000,
        });
        if (!response.ok) {
            (0, logger_1.log)(`Update check failed: ${response.status} ${response.statusText}`, 'WARN');
            return null;
        }
        const contentType = response.headers.get('content-type') || '';
        if (contentType.includes('text/html')) {
            (0, logger_1.log)(`Update check returned HTML instead of JSON`, 'WARN');
            return null;
        }
        const versionInfo = await response.json();
        const currentVersion = getCurrentVersion();
        (0, logger_1.log)(`Current: ${currentVersion}, Latest: ${versionInfo.version}`);
        if (compareVersions(versionInfo.version, currentVersion) > 0) {
            (0, logger_1.log)(`✅ Update available: ${versionInfo.version}`);
            (0, logger_1.log)(`   Release notes: ${versionInfo.releaseNotes}`);
            (0, logger_1.log)(`   Force update: ${versionInfo.forceUpdate}`);
            return versionInfo;
        }
        (0, logger_1.log)('Already running the latest version');
        return null;
    }
    catch (error) {
        (0, logger_1.log)(`Error checking for updates: ${error}`, 'ERROR');
        return null;
    }
}
/**
 * Download update ZIP file
 */
async function downloadUpdate(versionInfo) {
    try {
        (0, logger_1.log)(`Downloading update from: ${versionInfo.downloadUrl}`);
        const response = await (0, node_fetch_1.default)(versionInfo.downloadUrl, {
            method: 'GET',
            redirect: 'follow',
            timeout: 300000, // 5 minute timeout for large files
        });
        if (!response.ok) {
            (0, logger_1.log)(`Download failed: ${response.status} ${response.statusText}`, 'ERROR');
            return false;
        }
        // Check content type - should be application/zip or octet-stream
        const contentType = response.headers.get('content-type') || '';
        if (contentType.includes('text/html')) {
            (0, logger_1.log)(`Download returned HTML instead of ZIP - possible redirect error`, 'ERROR');
            return false;
        }
        // Get the response as buffer
        const buffer = await response.buffer();
        (0, logger_1.log)(`Downloaded ${buffer.length} bytes`);
        // Write to file
        ensureUpdatesDir();
        fs.writeFileSync(UPDATE_ZIP_PATH, buffer);
        (0, logger_1.log)(`Saved to: ${UPDATE_ZIP_PATH}`);
        // Verify hash
        const actualHash = calculateFileHash(UPDATE_ZIP_PATH);
        const expectedHash = versionInfo.sha256.toLowerCase();
        (0, logger_1.log)(`Expected SHA256: ${expectedHash}`);
        (0, logger_1.log)(`Actual SHA256:   ${actualHash}`);
        if (actualHash !== expectedHash) {
            (0, logger_1.log)(`❌ SHA256 MISMATCH - file corrupted or tampered!`, 'ERROR');
            fs.unlinkSync(UPDATE_ZIP_PATH);
            return false;
        }
        (0, logger_1.log)(`✅ SHA256 verified successfully`);
        // Write update info JSON for the scheduled task
        fs.writeFileSync(UPDATE_INFO_PATH, JSON.stringify(versionInfo, null, 2));
        (0, logger_1.log)(`Wrote update info to: ${UPDATE_INFO_PATH}`);
        return true;
    }
    catch (error) {
        (0, logger_1.log)(`Download error: ${error}`, 'ERROR');
        return false;
    }
}
/**
 * Trigger the scheduled task to perform the update
 */
function triggerUpdateTask() {
    try {
        (0, logger_1.log)(`Triggering scheduled task: ${SCHEDULED_TASK_NAME}`);
        // Run the scheduled task
        const result = (0, child_process_1.execSync)(`schtasks /Run /TN "${SCHEDULED_TASK_NAME}"`, {
            encoding: 'utf-8',
            timeout: 30000,
        });
        (0, logger_1.log)(`Task triggered: ${result.trim()}`);
        (0, logger_1.log)('=====================================');
        (0, logger_1.log)('UPDATE TASK TRIGGERED SUCCESSFULLY');
        (0, logger_1.log)('=====================================');
        (0, logger_1.log)('The scheduled task will:');
        (0, logger_1.log)('  1. Stop this service');
        (0, logger_1.log)('  2. Update files');
        (0, logger_1.log)('  3. Restart service');
        (0, logger_1.log)('This process will exit shortly...');
        (0, logger_1.log)('=====================================');
        return true;
    }
    catch (error) {
        (0, logger_1.log)(`Failed to trigger update task: ${error.message}`, 'ERROR');
        // Check if task exists
        try {
            (0, child_process_1.execSync)(`schtasks /Query /TN "${SCHEDULED_TASK_NAME}"`, { encoding: 'utf-8' });
            (0, logger_1.log)('Task exists but failed to run', 'ERROR');
        }
        catch {
            (0, logger_1.log)(`Task "${SCHEDULED_TASK_NAME}" not found - was Install-Service.bat run?`, 'ERROR');
        }
        return false;
    }
}
/**
 * Perform the full update process
 */
async function performUpdate(versionInfo) {
    // Check retry limit
    const failedAttempts = getFailedAttempts();
    if (failedAttempts >= MAX_RETRY_ATTEMPTS) {
        (0, logger_1.log)('=====================================', 'ERROR');
        (0, logger_1.log)('❌ MANUAL INTERVENTION REQUIRED', 'ERROR');
        (0, logger_1.log)('=====================================', 'ERROR');
        (0, logger_1.log)(`Update failed ${failedAttempts} times`, 'ERROR');
        (0, logger_1.log)('Please manually update the agent', 'ERROR');
        (0, logger_1.log)('=====================================', 'ERROR');
        return false;
    }
    (0, logger_1.log)('=====================================');
    (0, logger_1.log)(`STARTING UPDATE TO ${versionInfo.version}`);
    (0, logger_1.log)('=====================================');
    (0, logger_1.log)(`Attempt: ${failedAttempts + 1}/${MAX_RETRY_ATTEMPTS}`);
    // Step 1: Download and verify
    const downloaded = await downloadUpdate(versionInfo);
    if (!downloaded) {
        incrementFailedAttempts();
        return false;
    }
    // Step 2: Trigger scheduled task
    const triggered = triggerUpdateTask();
    if (!triggered) {
        incrementFailedAttempts();
        // Clean up downloaded files
        try {
            fs.unlinkSync(UPDATE_ZIP_PATH);
            fs.unlinkSync(UPDATE_INFO_PATH);
        }
        catch { }
        return false;
    }
    // Success - task will handle the rest
    // Reset failed attempts will happen in next version's startup
    return true;
}
/**
 * Start auto-update checker
 */
function startAutoUpdater(apiUrl) {
    (0, logger_1.log)('=====================================');
    (0, logger_1.log)('Auto-updater started (Scheduled Task Mode)');
    (0, logger_1.log)('=====================================');
    (0, logger_1.log)(`Check interval: ${UPDATE_CHECK_INTERVAL / 1000 / 60 / 60} hours`);
    (0, logger_1.log)(`Max retry attempts: ${MAX_RETRY_ATTEMPTS}`);
    (0, logger_1.log)(`Updates directory: ${UPDATES_DIR}`);
    (0, logger_1.log)(`Scheduled task: ${SCHEDULED_TASK_NAME}`);
    (0, logger_1.log)('-------------------------------------');
    // Ensure updates directory exists
    ensureUpdatesDir();
    // Initialize version file if needed
    const currentVersion = getCurrentVersion();
    const versionFile = getVersionFilePath();
    if (!fs.existsSync(versionFile)) {
        saveVersion(currentVersion, 0);
    }
    // Log retry state
    const failedAttempts = getFailedAttempts();
    if (failedAttempts > 0) {
        (0, logger_1.log)(`⚠️  Previous update failures: ${failedAttempts}/${MAX_RETRY_ATTEMPTS}`, 'WARN');
    }
    // Check for updates on startup (after 1 minute delay)
    setTimeout(async () => {
        (0, logger_1.log)('Running startup update check...');
        const updateInfo = await checkForUpdate(apiUrl);
        if (updateInfo) {
            await performUpdate(updateInfo);
        }
    }, 60000);
    // Check for updates every 6 hours
    setInterval(async () => {
        (0, logger_1.log)('Running scheduled update check...');
        const updateInfo = await checkForUpdate(apiUrl);
        if (updateInfo) {
            await performUpdate(updateInfo);
        }
    }, UPDATE_CHECK_INTERVAL);
    (0, logger_1.log)(`✅ Auto-updater initialized`);
}
//# sourceMappingURL=auto-updater.js.map